/***********************************************************************
 * \file: util_hash.h
 * 
 * \version: $Revision: 1.9 $
 * 
 * \release: $Name: UTIL_r2_E05 $
 *
 * Interface for hash table with chaining.
 * 
 * \component: Utility
 * 
 * \author T. Polle / ADIT / SWG / tpolle(o)de.adit-jv.com
 * 
 * \copyright: (c) 2003 - 2008 ADIT Corporation 
 *
 ***********************************************************************/
#ifndef UTIL_HASH_H
#define UTIL_HASH_H

#ifdef __cplusplus
extern "C" {
#endif

#include <adit_typedef.h>
/**
 * \ingroup utility
 * \defgroup hash Hash Table
 *
 * Provides util_hash tables.
 * \{
 */

/**
 * Hash table.
 */
typedef struct UTIL_hash_table_t UTIL_hash_table_t;

/**
 * Key of a hash table element.
 */
typedef struct
{
  VP   key;
  U32   key_len;
  VP   value;
} UTIL_hash_elem_t;

/**
 * Comparison function for hash table elements.
 * \returns  \li TRUE,  if the keys are equal,
 *           \li FALSE, if the keys are not equal.
 */
typedef BOOL (*UTIL_hash_cmp_fn)(UTIL_hash_elem_t* first,
                                 UTIL_hash_elem_t* second);

/**
 * Create a hash table.
 *
 * Allocates memory to accommodate a number of elements specified by
 * \a nel.
 *
 * \param nel   Number of maximal hash elements to be accommodated.
 * \param func  Comparison function for hash elements.
 *
 * \returns  \li Hash table, if the memory could be allocated.
 *           \li NULL,       if the memory could not be allocated.
 */
IMPORT UTIL_hash_table_t* UTIL_cre_hsh(U32                     nel,
                                       const UTIL_hash_cmp_fn func);

/**
 * Clear the contents of the hash table.
 *
 * All hash table elements are removed from the hash table. The memory
 * allocated is kept.
 *
 * \param table  Hash table.
 */
IMPORT void UTIL_clr_hsh(UTIL_hash_table_t* table);

/**
 * Delete the hash table.
 *
 * Memory allocated is freed as well.
 *
 * \param table  Hash table.
 *
 * \returns  E_OK
 */
IMPORT S32 UTIL_del_hsh(UTIL_hash_table_t* table);

/**
 * Search hash table \a table for \a elem.
 *
 * \param table  Hash table.
 * \param elem   Hash table element.
 *
 * \returns  \li Element, if found.
 *           \li NULL,    if not found.
 */
IMPORT UTIL_hash_elem_t* UTIL_sea_hsh(UTIL_hash_table_t* table,
                                      UTIL_hash_elem_t*  elem);

/**
 * Add a key to the hash table \a table.
 *
 * \a elem holds the key of the element to be added.
 *
 * \param table  Hash table.
 * \param elem   Key of the element to be added.
 *
 * \returns  \li Element, if the key \a elem was added,
 *           \li NULL,    if no more element is available.
 */
IMPORT UTIL_hash_elem_t* UTIL_add_hsh(UTIL_hash_table_t* table,
                                      UTIL_hash_elem_t*  elem);

/**
 * Search hash table \a table for \a elem.
 *
 * \param table  Hash table.
 * \param elem   Hash table element.
 *
 * \returns  E_OK
 */
IMPORT S32 UTIL_rem_hsh(UTIL_hash_table_t* table,
                       UTIL_hash_elem_t*  elem);

/**
 * \}
 */

#ifdef __cplusplus
};
#endif

#endif /* UTIL_HASH_H */
